/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.fs.gcs;

import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystem;
import com.google.cloud.hadoop.fs.gcs.GoogleHadoopFileSystemConfiguration;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageFileSystem;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.UriPaths;
import com.google.cloud.hadoop.gcsio.cooplock.CoopLockOperationDao;
import com.google.cloud.hadoop.gcsio.cooplock.CoopLockRecord;
import com.google.cloud.hadoop.gcsio.cooplock.CoopLockRecordsDao;
import com.google.cloud.hadoop.gcsio.cooplock.CooperativeLockingOptions;
import com.google.cloud.hadoop.gcsio.cooplock.DeleteOperation;
import com.google.cloud.hadoop.gcsio.cooplock.RenameOperation;
import com.google.cloud.hadoop.gcsio.cooplock.RenameOperationLogRecord;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.flogger.GoogleLogger;
import com.google.common.io.ByteSource;
import com.google.gson.Gson;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;

class CoopLockFsckRunner {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private static final Gson GSON = CoopLockRecordsDao.createGson();
    private final Instant operationExpirationInstant = Instant.now();
    private final String bucketName;
    private final String command;
    private final String fsckOperationId;
    private final GoogleHadoopFileSystem ghfs;
    private final GoogleCloudStorageFileSystem gcsFs;
    private final GoogleCloudStorageImpl gcs;
    private final CooperativeLockingOptions options;
    private final CoopLockRecordsDao lockRecordsDao;
    private final CoopLockOperationDao lockOperationDao;

    public CoopLockFsckRunner(Configuration conf, URI bucketUri, String command, String operationId) throws IOException {
        conf.setBoolean(GoogleHadoopFileSystemConfiguration.GCS_COOPERATIVE_LOCKING_ENABLE.getKey(), false);
        this.bucketName = bucketUri.getAuthority();
        this.command = command;
        this.fsckOperationId = operationId;
        this.ghfs = (GoogleHadoopFileSystem)FileSystem.get(bucketUri, conf);
        this.gcsFs = this.ghfs.getGcsFs();
        this.gcs = (GoogleCloudStorageImpl)this.gcsFs.getGcs();
        this.options = this.gcs.getOptions().getCooperativeLockingOptions();
        this.lockRecordsDao = new CoopLockRecordsDao(this.gcs);
        this.lockOperationDao = new CoopLockOperationDao(this.gcs);
    }

    public int run() throws IOException {
        Set<CoopLockRecord> lockedOperations = this.lockRecordsDao.getLockedOperations(this.bucketName);
        if (lockedOperations.isEmpty()) {
            ((GoogleLogger.Api)logger.atInfo()).log("No expired operation locks");
            return 0;
        }
        Map<FileStatus, CoopLockRecord> expiredOperations = lockedOperations.stream().map(this::getOperationLockIfExpired).filter(Optional::isPresent).map(Optional::get).filter(this::checkLogExistsAndUnlockOtherwise).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        if ("--check".equals(this.command)) {
            return 0;
        }
        if (!"all".equals(this.fsckOperationId)) {
            Optional<Map.Entry> operationEntry = expiredOperations.entrySet().stream().filter(e -> ((CoopLockRecord)e.getValue()).getOperationId().equals(this.fsckOperationId)).findAny();
            Preconditions.checkArgument(operationEntry.isPresent(), "%s operation not found", (Object)this.fsckOperationId);
            expiredOperations = ImmutableMap.of(operationEntry.get().getKey(), operationEntry.get().getValue());
        }
        expiredOperations.forEach((operationStatus, operationRecord) -> {
            long start = System.currentTimeMillis();
            try {
                this.repairOperation((FileStatus)operationStatus, (CoopLockRecord)operationRecord);
                ((GoogleLogger.Api)logger.atInfo()).log("Operation %s successfully %s in %dms", operationStatus.getPath(), "--rollForward".equals(this.command) ? "rolled forward" : "rolled back", System.currentTimeMillis() - start);
            }
            catch (Exception e) {
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atSevere()).withCause(e)).log("Operation %s failed to %s in %dms", operationStatus.getPath(), "--rollForward".equals(this.command) ? "roll forward" : "roll back", System.currentTimeMillis() - start);
            }
        });
        return 0;
    }

    private void repairOperation(FileStatus operationStatus, CoopLockRecord operationRecord) throws IOException, URISyntaxException {
        switch (operationRecord.getOperationType()) {
            case DELETE: {
                this.repairDeleteOperation(operationStatus, operationRecord);
                return;
            }
            case RENAME: {
                this.repairRenameOperation(operationStatus, operationRecord);
                return;
            }
        }
        throw new IllegalStateException(String.format("Unknown %s operation type: %s", new Object[]{operationRecord.getOperationId(), operationRecord.getOperationType()}));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void repairDeleteOperation(FileStatus operationStatus, CoopLockRecord operationRecord) throws IOException, URISyntaxException {
        if ("--rollBack".equals(this.command)) {
            ((GoogleLogger.Api)logger.atInfo()).log("Rolling back delete operations (%s) not supported, skipping.", operationStatus.getPath());
            return;
        }
        ((GoogleLogger.Api)logger.atInfo()).log("Repairing FS after %s delete operation.", operationStatus.getPath());
        DeleteOperation operation = this.getOperation(operationStatus, DeleteOperation.class);
        this.lockRecordsDao.relockOperation(this.bucketName, operationRecord);
        Future<?> lockUpdateFuture = this.lockOperationDao.scheduleLockUpdate(operationRecord.getOperationId(), new URI(operationStatus.getPath().toString()), DeleteOperation.class, (o, updateInstant) -> o.setLockExpiration(updateInstant.plusMillis(this.options.getLockExpirationTimeoutMilli())));
        try {
            List<String> loggedResources = this.getOperationLog(operationStatus, l -> l);
            this.deleteResource(operation.getResource(), loggedResources);
            this.lockRecordsDao.unlockPaths(operationRecord.getOperationId(), StorageResourceId.fromStringPath(operation.getResource()));
        }
        finally {
            lockUpdateFuture.cancel(false);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void repairRenameOperation(FileStatus operationStatus, CoopLockRecord operationRecord) throws IOException, URISyntaxException {
        RenameOperation operation = this.getOperation(operationStatus, RenameOperation.class);
        this.lockRecordsDao.relockOperation(this.bucketName, operationRecord);
        Future<?> lockUpdateFuture = this.lockOperationDao.scheduleLockUpdate(operationRecord.getOperationId(), new URI(operationStatus.getPath().toString()), RenameOperation.class, (o, updateInstant) -> o.setLockExpiration(updateInstant.plusMillis(this.options.getLockExpirationTimeoutMilli())));
        try {
            LinkedHashMap loggedResources = this.getOperationLog(operationStatus, l -> GSON.fromJson((String)l, RenameOperationLogRecord.class)).stream().collect(Collectors.toMap(RenameOperationLogRecord::getSrc, RenameOperationLogRecord::getDst, (e1, e2) -> {
                throw new RuntimeException(String.format("Found entries with duplicate keys: %s and %s", e1, e2));
            }, LinkedHashMap::new));
            if (operation.getCopySucceeded()) {
                if ("--rollBack".equals(this.command)) {
                    this.deleteAndRenameToRepairRenameOperation(operationStatus, operationRecord, operation.getDstResource(), new ArrayList<String>(loggedResources.values()), operation.getSrcResource(), "source", new ArrayList<String>(loggedResources.keySet()), false);
                } else {
                    this.deleteToRepairRenameOperation(operationStatus, operation.getSrcResource(), "source", loggedResources.keySet());
                }
            } else if ("--rollBack".equals(this.command)) {
                this.deleteToRepairRenameOperation(operationStatus, operation.getDstResource(), "destination", loggedResources.values());
            } else {
                this.deleteAndRenameToRepairRenameOperation(operationStatus, operationRecord, operation.getSrcResource(), new ArrayList<String>(loggedResources.keySet()), operation.getDstResource(), "destination", new ArrayList<String>(loggedResources.values()), true);
            }
            this.lockRecordsDao.unlockPaths(operationRecord.getOperationId(), StorageResourceId.fromStringPath(operation.getSrcResource()), StorageResourceId.fromStringPath(operation.getDstResource()));
        }
        finally {
            lockUpdateFuture.cancel(false);
        }
    }

    private void deleteToRepairRenameOperation(FileStatus operationLock, String operationResource, String deleteResourceType, Collection<String> loggedResources) throws IOException {
        ((GoogleLogger.Api)logger.atInfo()).log("Repairing FS after %s rename operation (deleting %s (%s)).", operationLock.getPath(), deleteResourceType, operationResource);
        this.deleteResource(operationResource, loggedResources);
    }

    private void deleteAndRenameToRepairRenameOperation(FileStatus operationLock, CoopLockRecord operation, String srcResource, List<String> loggedSrcResources, String dstResource, String dstResourceType, List<String> loggedDstResources, boolean copySucceeded) throws IOException {
        ((GoogleLogger.Api)logger.atInfo()).log("Repairing FS after %s rename operation (deleting %s (%s) and renaming (%s -> %s)).", operationLock.getPath(), dstResourceType, dstResource, srcResource, dstResource);
        this.deleteResource(dstResource, loggedDstResources);
        this.gcs.copy(this.bucketName, CoopLockFsckRunner.toNames(loggedSrcResources), this.bucketName, CoopLockFsckRunner.toNames(loggedDstResources));
        this.lockOperationDao.checkpointRenameOperation(this.bucketName, operation.getOperationId(), operation.getOperationTime(), copySucceeded);
        this.deleteResource(srcResource, loggedSrcResources);
    }

    private static List<String> toNames(List<String> resources) {
        return resources.stream().map(r -> StorageResourceId.fromStringPath(r).getObjectName()).collect(Collectors.toList());
    }

    private Optional<Map.Entry<FileStatus, CoopLockRecord>> getOperationLockIfExpired(CoopLockRecord operation) {
        try {
            return this.getOperationLockIfExpiredChecked(operation);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to check if %s operation expired", operation), e);
        }
    }

    private Optional<Map.Entry<FileStatus, CoopLockRecord>> getOperationLockIfExpiredChecked(CoopLockRecord operationRecord) throws IOException {
        String globPath = "_lock/*" + operationRecord.getOperationId() + "*.lock";
        URI globUri = UriPaths.fromStringPathComponents(this.bucketName, globPath, false);
        FileStatus[] operationLock = this.ghfs.globStatus(new Path(globUri));
        Preconditions.checkState(operationLock.length < 2, "operation %s should not have more than one lock file", (Object)operationRecord.getOperationId());
        if (operationLock.length == 0) {
            this.unlockOperationIfNecessary(operationRecord, "lock");
            return Optional.empty();
        }
        FileStatus operationStatus = operationLock[0];
        if (operationRecord.getLockExpiration().isBefore(this.operationExpirationInstant) && this.getRenewedLockExpiration(operationStatus, operationRecord).isBefore(this.operationExpirationInstant)) {
            ((GoogleLogger.Api)logger.atInfo()).log("Operation %s expired.", operationStatus.getPath());
            return Optional.of(new AbstractMap.SimpleEntry<FileStatus, CoopLockRecord>(operationStatus, operationRecord));
        }
        ((GoogleLogger.Api)logger.atInfo()).log("Operation %s not expired.", operationStatus.getPath());
        return Optional.empty();
    }

    private boolean checkLogExistsAndUnlockOtherwise(Map.Entry<FileStatus, CoopLockRecord> operation) {
        try {
            return this.checkLogExistsAndUnlockOtherwiseChecked(operation);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to check if %s operation expired", operation.getValue()), e);
        }
    }

    private boolean checkLogExistsAndUnlockOtherwiseChecked(Map.Entry<FileStatus, CoopLockRecord> operation) throws IOException {
        CoopLockRecord operationRecord = operation.getValue();
        String globPath = "_lock/*" + operationRecord.getOperationId() + "*.log";
        URI globUri = UriPaths.fromStringPathComponents(this.bucketName, globPath, false);
        FileStatus[] operationLog = this.ghfs.globStatus(new Path(globUri));
        Preconditions.checkState(operationLog.length < 2, "operation %s should not have more than one log file", (Object)operationRecord.getOperationId());
        if (operationLog.length == 0) {
            this.unlockOperationIfNecessary(operationRecord, "log");
            return false;
        }
        return true;
    }

    private void unlockOperationIfNecessary(CoopLockRecord operationRecord, String fileType) throws IOException {
        if (!"--check".equals(this.command) && ("all".equals(this.fsckOperationId) || this.fsckOperationId.equals(operationRecord.getOperationId()))) {
            ((GoogleLogger.Api)logger.atInfo()).log("Operation %s for %s resources doesn't have %s file, unlocking", operationRecord.getOperationId(), operationRecord.getResources(), fileType);
            StorageResourceId[] lockedResources = (StorageResourceId[])operationRecord.getResources().stream().map(resource -> new StorageResourceId(this.bucketName, (String)resource)).toArray(StorageResourceId[]::new);
            this.lockRecordsDao.unlockPaths(operationRecord.getOperationId(), lockedResources);
        } else {
            ((GoogleLogger.Api)logger.atInfo()).log("Operation %s for %s resources doesn't have %s file, skipping", operationRecord.getOperationId(), operationRecord.getResources(), fileType);
        }
    }

    private void deleteResource(String resource, Collection<String> loggedResources) throws IOException {
        Path lockedResource = new Path(resource);
        Set allObjects = Arrays.stream(this.ghfs.listStatus(lockedResource)).map(s2 -> s2.getPath().toString()).collect(Collectors.toSet());
        ArrayList<StorageResourceId> objectsToDelete = new ArrayList<StorageResourceId>(loggedResources.size());
        for (String loggedObject : loggedResources) {
            if (!allObjects.contains(loggedObject)) continue;
            objectsToDelete.add(StorageResourceId.fromStringPath(loggedObject));
        }
        GoogleCloudStorage gcs = this.ghfs.getGcsFs().getGcs();
        gcs.deleteObjects(objectsToDelete);
        allObjects.removeAll(loggedResources);
        if (allObjects.isEmpty() && this.ghfs.exists(lockedResource)) {
            this.ghfs.delete(lockedResource, false);
        }
    }

    private Instant getRenewedLockExpiration(FileStatus operationStatus, CoopLockRecord operationRecord) throws IOException {
        switch (operationRecord.getOperationType()) {
            case DELETE: {
                return this.getOperation(operationStatus, DeleteOperation.class).getLockExpiration();
            }
            case RENAME: {
                return this.getOperation(operationStatus, RenameOperation.class).getLockExpiration();
            }
        }
        throw new IllegalStateException(String.format("Unknown %s operation type: %s", new Object[]{operationStatus.getPath(), operationRecord.getOperationType()}));
    }

    private <T> T getOperation(final FileStatus operationStatus, Class<T> clazz) throws IOException {
        ByteSource operationByteSource = new ByteSource(){

            @Override
            public InputStream openStream() throws IOException {
                return CoopLockFsckRunner.this.ghfs.open(operationStatus.getPath());
            }
        };
        String operationContent = operationByteSource.asCharSource(StandardCharsets.UTF_8).read();
        return GSON.fromJson(operationContent, clazz);
    }

    private <T> List<T> getOperationLog(FileStatus operationStatus, Function<String, T> logRecordFn) throws IOException {
        ArrayList<T> log = new ArrayList<T>();
        Path operationLog = new Path(operationStatus.getPath().toString().replace(".lock", ".log"));
        try (BufferedReader in = new BufferedReader(new InputStreamReader((InputStream)this.ghfs.open(operationLog), StandardCharsets.UTF_8));){
            String line;
            while ((line = in.readLine()) != null) {
                log.add(logRecordFn.apply(line));
            }
        }
        return log;
    }
}

