/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.seatunnel.server.common;

import lombok.extern.slf4j.Slf4j;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Date;

@Slf4j
public class DateUtils {
    public static final String DEFAULT_DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
    public static final String DEFAULT_DATETIME_FORMAT_WITH_TIMEZONE =
            "yyyy-MM-dd'T'HH:mm:ss.SSSZZZ";

    /**
     * parse Date to String Date, use default datetime format 'yyyy-MM-dd HH:mm:ss'
     *
     * @param date which need been format to String date
     * @return String date
     */
    public static String format(Date date) {
        return format(date2LocalDateTime(date), DEFAULT_DATETIME_FORMAT);
    }

    public static String format(Date date, String format) {
        return format(date2LocalDateTime(date), format);
    }

    public static String format(LocalDateTime localDateTime, String format) {
        return localDateTime.format(DateTimeFormatter.ofPattern(format));
    }

    private static LocalDateTime date2LocalDateTime(Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
    }

    /**
     * parse String date to Date, use default datetime format 'yyyy-MM-dd HH:mm:ss'
     *
     * @param date which need been parse to Date
     * @return Date
     */
    public static Date parse(String date) {
        LocalDateTime localDateTime =
                LocalDateTime.parse(date, DateTimeFormatter.ofPattern(DEFAULT_DATETIME_FORMAT));
        return localDateTime2Date(localDateTime);
    }

    public static Date parse(String date, String format) {
        LocalDateTime localDateTime =
                LocalDateTime.parse(date, DateTimeFormatter.ofPattern(format));
        return localDateTime2Date(localDateTime);
    }

    private static Date localDateTime2Date(LocalDateTime localDateTime) {
        Instant instant = localDateTime.atZone(ZoneId.systemDefault()).toInstant();
        return Date.from(instant);
    }
}
