package org.apache.stratum.component;

/*
 * Copyright 2001-2005 The Apache Software Foundation or its licensors,
 * as applicable.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;
import java.util.List;

import org.apache.commons.configuration.CompositeConfiguration;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.log4j.Category;
import org.apache.stratum.lifecycle.Configurable;
import org.apache.stratum.lifecycle.Initializable;

/**
 * Loader for Components implementing the lifecyle Interfaces. NOTE: This class is in its infancy and will more than likely change.
 *
 * @author <a href="mailto:eric NOSPAM dobbse.net">Eric Dobbs </a>
 * @author <a href="mailto:mpoeschl@marmot.at">Martin Poeschl </a>
 * @version $Id: ComponentLoader.java 264191 2005-08-29 18:07:52Z henning $
 */
public class ComponentLoader
{
    /** Log4j category used for logging. NOTE: we should change this to use the commons-logging API. */
    private static Category log = Category.getInstance(ComponentLoader.class);

    /** Component tag used in Configurations */
    private static String COMPONENT = "component";

    /** Class name tag used in Configurations */
    private static String CLASSNAME = "classname";

    /** Extension used for Configuration files. */
    private static String CONFIG = "config";

    /** Extension used for Additional properties. */
    private static String PROPERTY = "property";

    /** Name tag used in Configurations */
    private static String NAME = "name";

    /** Configuration used by this ComponentLoader. */
    private Configuration configuration;

    /**
     * Constructor
     *
     * @param configuration
     */
    public ComponentLoader(Configuration configuration)
    {
        this.configuration = configuration;
    }

    /**
     * Set the configuration for this ComponentLoader
     *
     * @param configuration Configuration
     */
    public void setConfiguration(Configuration configuration)
    {
        this.configuration = configuration;
    }

    /**
     * Support method for testing the constructor
     *
     * @return the configuration
     */
    public Configuration getConfiguration()
    {
        return configuration;
    }

    /**
     * <p>
     * Load all the components listed in the ComponentLoader's configuration. Log any errors, but throw no exceptions if components
     * cannot be loaded.
     * </p>
     * Configuration notes:<br>
     * Components are identified in the properties file as follows: <br>
     * <code> component.name=NAME component.NAME.classname = com.mycompany.components.SomeComponent component.NAME.config    =
     * path/to/SomeComponent.properties </code>
     *
     * @return an array of loaded components
     */
    public Object [] load()
    {
        List components = configuration.getList(COMPONENT + '.' + NAME);
        Object [] loadedComponents = new Object[components.size()];

        String componentName;
        String componentClassName;
        String componentConfig;
        Configuration componentAdditionalConfig;

        for (int i = 0; i < components.size(); i++)
        {
            componentName = (String) components.get(i);
            componentClassName = getComponentClassname(componentName);
            componentConfig = getComponentConfigFile(componentName);
            componentAdditionalConfig = getComponentAdditionalConfig(componentName);

            log.info("loading component: name=" + componentName + " class=" + componentClassName + " config=" + componentConfig);

            loadedComponents[i] = loadComponent(componentClassName, componentConfig, componentAdditionalConfig);
        }

        return loadedComponents;
    }

    /**
     * load the given component, configure it with the given config file, and initialize it. <br>
     * The component must implement the <code>Initializable</code> and <code>Configurable</code> interfaces.
     *
     * @param className the String class name of the component to load
     * @param configFile the String path name of the component's config file
     * @param additionalConfig TODO: DOCUMENT ME!
     *
     * @return the loaded component or null if it failed to load
     *
     * @see Initializable
     * @see Configurable
     */
    public Object loadComponent(String className, String configFile, Configuration additionalConfig)
    {
        Object component = null;

        if (log.isDebugEnabled())
        {
            log.debug("attempting to load '" + className + "' with the config file '" + configFile + "'.");
        }

        try
        {
            component = Class.forName(className).newInstance();

            // configure component using the given config file
            CompositeConfiguration config = new CompositeConfiguration();
            config.addConfiguration(new PropertiesConfiguration(configFile));
            config.addConfiguration(additionalConfig);
            ((Configurable) component).configure(config);

            // initialize component
            ((Initializable) component).initialize();

            if (log.isDebugEnabled())
            {
                log.debug("good news! " + className + " successfully configured and initialized");
            }
        }
        catch (IOException ioe)
        {
            log.error(className + " could not be configured with file '" + configFile + "'.", ioe);
        }
        catch (Exception e)
        {
            log.error(className + " could not be initialized!", e);
        }

        return component;
    }

    /**
     * <p>
     * Get the component's classname as defined in the ComponentLoader configuration.
     * </p>
     *
     * <p>
     * Example property: <br/> component.NAME.classname=com.mycompany.components.MyComponent
     * </p>
     *
     * @param name the String NAME of the component in the classfile
     *
     * @return the configured classname
     */
    private String getComponentClassname(String name)
    {
        return configuration.getString(COMPONENT + '.' + name + '.' + CLASSNAME);
    }

    /**
     * <p>
     * Get the component's config file as defined in the ComponentLoader configuration.
     * </p>
     *
     * <p>
     * Example property: <br/>component.NAME.config=path/to/your/config
     * </p>
     *
     * @param name the String NAME of the component in the classfile
     *
     * @return the configured config file
     */
    private String getComponentConfigFile(String name)
    {
        return configuration.getString(COMPONENT + '.' + name + '.' + CONFIG);
    }

    /**
     * <p>
     * Get any additional configuration properties from the master configuration.
     * </p>
     *
     * <p>
     * Example property: <br/>
     * component.NAME.property.key1==value1 <br/>
     * component.NAME.property.key2==value2 <br/>
     * component.NAME.property.key3==value3 <br/>
     * </p>
     *
     * @param name the String NAME of the component in the classfile
     *
     * @return the extra properties as a Configuration object
     */
    private Configuration getComponentAdditionalConfig(String name)
    {
        return configuration.subset(COMPONENT + '.' + name + '.' + PROPERTY);
    }
}
