package PROP::Conditions::Local;

use strict;
use Hash::Util qw/lock_keys/;
use PROP::Constants;
use PROP::Util;

sub new {
    my ($class, $expressions, $bindings) = @_;
    my $self = bless({}, $class);

    my $err_msg;

    $err_msg = "was expecting array ref of bindings as second argument"
	unless ref $bindings eq 'ARRAY';

    $err_msg = "was expecting array ref of expressions as first argument"
	unless ref $expressions eq 'ARRAY';
	
    die new PROP::Exception::IllegalArgument($err_msg) if $err_msg;

    $self->{-expressions} = $expressions;
    $self->{-bindings}    = $bindings;

    lock_keys(%$self) if DEBUG;

    return $self;
}

sub get_expressions {
    my ($self) = @_;
    return @{$self->{-expressions}};
}

sub push_expression {
    my ($self, $expression) = @_;
    push(@{$self->{-expressions}}, $expression);
}

sub get_bindings {
    my ($self) = @_;
    return @{$self->{-bindings}};
}

sub push_binding {
    my ($self, $binding) = @_;
    push(@{$self->{-bindings}}, $binding);
}

1;

=head1 Name

PROP::Conditions::Local

=head1 Synopsis

 $conds = new PROP::Conditions::Local($expressions, $bindings);

=head1 Methods

=over

=item new

 $conds = new PROP::Conditions::Local($expressions, $bindings);

This method constructs and returns a new PROP::Conditions::Local
object, where $expressions is an array reference of strings that
represent conditional expressions, e.g. 'foo < ?', and $bindings is an
array reference of bindings for variables (?) in the conditional
expressions.

=item get_expressions

 @expressions = $conds->get_expressions();

This method returns a list that contains the conditional expressions
that this object holds.

=item get_bindings

 @bindings = $conds->get_bindings();

This method returns a list that contains the conditional bindings
that this object holds.

=item push_expression

 $conds->push_expression($expression);

This method adds an expression to the end of the list of expressions
held by this object.

=item push_binding

 $conds->push_binding($binding);

This method adds a binding to the end of the list of bindings held by
this object.

=back

=head1 Author

Andrew Gibbs (awgibbs@awgibbs.com,andrew.gibbs@nist.gov)

=head1 Legalese

This software was developed at the National Institute of Standards and
Technology by employees of the Federal Government in the course of
their official duties. Pursuant to title 17 Section 105 of the United
States Code this software is not subject to copyright protection and
is in the public domain. PROP is an experimental system. NIST
assumes no responsibility whatsoever for its use by other parties, and
makes no guarantees, expressed or implied, about its quality,
reliability, or any other characteristic. We would appreciate
acknowledgement if the software is used.  This software can be
redistributed and/or modified freely provided that any derivative
works bear some notice that they are derived from it, and any modified
versions bear some notice that they have been modified.
