/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz (peter.penz@gmx.at)                  *
 *   Copyright (C) 2008 by Simon St. James (kdedevel@etotheipiplusone.com) *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
 ***************************************************************************/

#ifndef DOLPHINDETAILSVIEW_H
#define DOLPHINDETAILSVIEW_H

#include <dolphinview.h>
#include <QTreeView>
#include <libdolphin_export.h>

class DolphinViewController;
class DolphinSortFilterProxyModel;
class ViewExtensionsFactory;

/**
 * @brief Represents the details view which shows the name, size,
 *        date, permissions, owner and group of an item.
 *
 * The width of the columns is automatically adjusted in a way
 * that full available width of the view is used by stretching the width
 * of the name column.
 */
class LIBDOLPHINPRIVATE_EXPORT DolphinDetailsView : public QTreeView
{
    Q_OBJECT

public:
    /**
     * @param parent                Parent widget.
     * @param dolphinViewController Allows the DolphinDetailsView to control the
     *                              DolphinView in a limited way.
     * @param viewModeController    Controller that is used by the DolphinView
     *                              to control the DolphinDetailsView. The DolphinDetailsView
     *                              only has read access to the controller.
     * @param model                 Directory that is shown.
     */
    explicit DolphinDetailsView(QWidget* parent,
                                DolphinViewController* dolphinViewController,
                                const ViewModeController* viewModeController,
                                DolphinSortFilterProxyModel* model);
    virtual ~DolphinDetailsView();

    /**
     * Returns a set containing the URLs of all expanded items.
     */
    QSet<KUrl> expandedUrls() const;

    virtual QRegion visualRegionForSelection(const QItemSelection& selection) const;

protected:
    virtual bool event(QEvent* event);
    virtual QStyleOptionViewItem viewOptions() const;
    virtual void contextMenuEvent(QContextMenuEvent* event);
    virtual void mousePressEvent(QMouseEvent* event);
    virtual void mouseMoveEvent(QMouseEvent* event);
    virtual void mouseReleaseEvent(QMouseEvent* event);
    virtual void startDrag(Qt::DropActions supportedActions);
    virtual void dragEnterEvent(QDragEnterEvent* event);
    virtual void dragLeaveEvent(QDragLeaveEvent* event);
    virtual void dragMoveEvent(QDragMoveEvent* event);
    virtual void dropEvent(QDropEvent* event);
    virtual void paintEvent(QPaintEvent* event);
    virtual void keyPressEvent(QKeyEvent* event);
    virtual void keyReleaseEvent(QKeyEvent* event);
    virtual void resizeEvent(QResizeEvent* event);
    virtual void wheelEvent(QWheelEvent* event);
    virtual void currentChanged(const QModelIndex& current, const QModelIndex& previous);
    virtual bool eventFilter(QObject* watched, QEvent* event);
    virtual QModelIndex indexAt (const QPoint& point) const;
    virtual QRect visualRect(const QModelIndex& index) const;
    virtual void setSelection(const QRect& rect, QItemSelectionModel::SelectionFlags command);
    virtual void scrollTo(const QModelIndex& index, ScrollHint hint = EnsureVisible);

private slots:
    /**
     * Sets the sort indicator section of the header view
     * corresponding to \a sorting.
     */
    void setSortIndicatorSection(DolphinView::Sorting sorting);

    /**
     * Sets the sort indicator order of the header view
     * corresponding to \a sortOrder.
     */
    void setSortIndicatorOrder(Qt::SortOrder sortOrder);

    /**
     * Synchronizes the sorting state of the Dolphin menu 'View -> Sort'
     * with the current state of the details view.
     * @param column Index of the current sorting column.
     */
    void synchronizeSortingState(int column);

    /**
     * Is invoked when the mouse cursor has entered an item. The controller
     * gets informed to emit the itemEntered() signal if the mouse cursor
     * is above the name column. Otherwise the controller gets informed
     * to emit the itemViewportEntered() signal (all other columns should
     * behave as viewport area).
     */
    void slotEntered(const QModelIndex& index);

    /**
     * Updates the destination \a destination from
     * the elastic band to the current mouse position and triggers
     * an update.
     */
    void updateElasticBand();

    /**
     * Returns the rectangle for the elastic band dependent from the
     * origin \a origin, the current destination
     * \a destination and the viewport position.
     */
    QRect elasticBandRect() const;

    void setZoomLevel(int level);

    void slotShowPreviewChanged();

    /**
     * Opens a context menu at the position \a pos and allows to
     * configure the visibility of the header columns and whether
     * expandable folders should be shown.
     */
    void configureSettings(const QPoint& pos);

    /**
     * Updates the visibilty state of columns and their order.
     */
    void updateColumnVisibility();

    /**
     * Saves order of the columns as global setting.
     */
    void saveColumnPositions();

    /**
     * Disables the automatical resizing of columns, if the user has resized the columns
     * with the mouse.
     */
    void slotHeaderSectionResized(int logicalIndex, int oldSize, int newSize);

    /**
     * Changes the alternating row colors setting depending from
     * the activation state \a active.
     */
    void slotActivationChanged(bool active);

    /**
     * Disables the automatical resizing of the columns. Per default all columns
     * are resized to use the maximum available width of the view as good as possible.
     */
    void disableAutoResizing();

    void requestActivation();

    void slotGlobalSettingsChanged(int category);

    /**
     * If the elastic band is currently shown, update the elastic band based on
     * the current mouse position and ensure that the selection is the set of items
     * intersecting it.
     */
    void updateElasticBandSelection();

    /**
     * If \a expandable is true, the details view acts as tree view.
     * The current expandable state is remembered in the settings.
     */
    void setFoldersExpandable(bool expandable);

    /**
     * These slots update the list of expanded items.
     */
    void slotExpanded(const QModelIndex& index);
    void slotCollapsed(const QModelIndex& index);

protected slots:

    virtual void rowsAboutToBeRemoved(const QModelIndex &parent, int start, int end);

private:
    /**
     * Removes the URLs corresponding to the children of \a index in the rows
     * between \a start and \a end inclusive from the set of expanded URLs.
     */
    void removeExpandedIndexes(const QModelIndex& parent, int start, int end);

    /**
     * Updates the size of the decoration dependent on the
     * icon size of the DetailsModeSettings. The controller
     * will get informed about possible zoom in/zoom out
     * operations.
     */
    void updateDecorationSize(bool showPreview);

    KFileItemDelegate::Information infoForColumn(int columnIndex) const;

    /**
     * Resizes all columns in a way to use the whole available width of the view.
     */
    void resizeColumns();

    /**
     * Returns true, if \a pos is within the expanding toggle of a tree.
     */
    bool isAboveExpandingToggle(const QPoint& pos) const;

    /**
     * Sets the maximum size available for editing in the delegate.
     */
    void adjustMaximumSizeForEditing(const QModelIndex& index);

private:
    bool m_autoResize : 1;        // if true, the columns are resized automatically to the available width
    bool m_expandingTogglePressed : 1;
    bool m_keyPressed : 1;        // true if a key is pressed currently; info used by currentChanged()
    bool m_useDefaultIndexAt : 1; // true, if QTreeView::indexAt() should be used
    bool m_ignoreScrollTo : 1;    // true if calls to scrollTo(...) should do nothing.

    DolphinViewController* m_dolphinViewController;
    const ViewModeController* m_viewModeController;
    ViewExtensionsFactory* m_extensionsFactory;
    QAction* m_expandableFoldersAction;

    // A set containing the URLs of all currently expanded folders.
    // We cannot use a QSet<QModelIndex> because a QModelIndex is not guaranteed to remain valid over time.
    // Also a QSet<QPersistentModelIndex> does not work as expected because it is not guaranteed that
    // subsequent expand/collapse events of the same file item will yield the same QPersistentModelIndex.
    QSet<KUrl> m_expandedUrls;

    QFont m_font;
    QSize m_decorationSize;

    QRect m_dropRect;

    struct ElasticBand
    {
        ElasticBand();

        // Elastic band origin and destination coordinates are relative to t
        // he origin of the view, not the viewport.
        bool show;
        QPoint origin;
        QPoint destination;

        // Optimization mechanisms for use with elastic band selection.
        // Basically, allow "incremental" updates to the selection based
        // on the last elastic band shape.
        QPoint lastSelectionOrigin;
        QPoint lastSelectionDestination;

        // If true, compute the set of selected elements from scratch (slower)
        bool ignoreOldInfo;

        // Edges of the filenames that are closest to the edges of oldSelectionRect.
        // Used to decide whether horizontal changes in the elastic band are likely
        // to require us to re-check which items are selected.
        int outsideNearestLeftEdge;
        int outsideNearestRightEdge;
        int insideNearestLeftEdge;
        int insideNearestRightEdge;
        // The set of items that were selected at the time this band was shown.
        // NOTE: Unless CTRL was pressed when the band was created, this is always empty.
        QItemSelection originalSelection;
    } m_band;
};

#endif
