// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#include "olap/rowset/segment_v2/bloom_filter_index_reader.h"

#include <gen_cpp/segment_v2.pb.h>
#include <glog/logging.h>

#include <memory>

#include "olap/rowset/segment_v2/bloom_filter.h"
#include "olap/types.h"
#include "util/debug_points.h"
#include "vec/columns/column.h"
#include "vec/common/string_ref.h"
#include "vec/data_types/data_type.h"
#include "vec/data_types/data_type_factory.hpp"

namespace doris::segment_v2 {
#include "common/compile_check_begin.h"
Status BloomFilterIndexReader::load(bool use_page_cache, bool kept_in_memory,
                                    OlapReaderStatistics* index_load_stats) {
    // TODO yyq: implement a new once flag to avoid status construct.
    return _load_once.call([this, use_page_cache, kept_in_memory, index_load_stats] {
        return _load(use_page_cache, kept_in_memory, index_load_stats);
    });
}

int64_t BloomFilterIndexReader::get_metadata_size() const {
    return sizeof(BloomFilterIndexReader) +
           (_bloom_filter_index_meta ? _bloom_filter_index_meta->ByteSizeLong() : 0);
}

Status BloomFilterIndexReader::_load(bool use_page_cache, bool kept_in_memory,
                                     OlapReaderStatistics* index_load_stats) {
    const IndexedColumnMetaPB& bf_index_meta = _bloom_filter_index_meta->bloom_filter();

    _bloom_filter_reader = std::make_unique<IndexedColumnReader>(_file_reader, bf_index_meta);
    RETURN_IF_ERROR(_bloom_filter_reader->load(use_page_cache, kept_in_memory, index_load_stats));
    update_metadata_size();
    return Status::OK();
}

Status BloomFilterIndexReader::new_iterator(std::unique_ptr<BloomFilterIndexIterator>* iterator,
                                            OlapReaderStatistics* index_load_stats) {
    DBUG_EXECUTE_IF("BloomFilterIndexReader::new_iterator.fail", {
        return Status::InternalError("new_iterator for bloom filter index failed");
    });
    *iterator = std::make_unique<BloomFilterIndexIterator>(this, index_load_stats);
    return Status::OK();
}

Status BloomFilterIndexIterator::read_bloom_filter(rowid_t ordinal,
                                                   std::unique_ptr<BloomFilter>* bf) {
    size_t num_to_read = 1;
    auto data_type = vectorized::DataTypeFactory::instance().create_data_type(
            _reader->type_info()->type(), 1, 0);
    auto column = data_type->create_column();

    RETURN_IF_ERROR(_bloom_filter_iter.seek_to_ordinal(ordinal));
    DCHECK(current_bloom_filter_index() == ordinal);
    size_t num_read = num_to_read;
    RETURN_IF_ERROR(_bloom_filter_iter.next_batch(&num_read, column));
    DCHECK(num_to_read == num_read);
    // construct bloom filter
    StringRef value = column->get_data_at(0);
    RETURN_IF_ERROR(
            BloomFilter::create(_reader->_bloom_filter_index_meta->algorithm(), bf, value.size));
    RETURN_IF_ERROR((*bf)->init(value.data, value.size,
                                _reader->_bloom_filter_index_meta->hash_strategy()));
    return Status::OK();
}

} // namespace doris::segment_v2
#include "common/compile_check_end.h"
