// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package repos

import (
	"net/http"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"
	"github.com/drone/drone/logger"

	"github.com/go-chi/chi"
)

// HandleRepair returns an http.HandlerFunc that processes http
// requests to repair the repository hooks and sync the repository
// details.
func HandleRepair(
	hooks core.HookService,
	repoz core.RepositoryService,
	repos core.RepositoryStore,
	users core.UserStore,
	link string,
) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		var (
			owner = chi.URLParam(r, "owner")
			name  = chi.URLParam(r, "name")
		)

		repo, err := repos.FindName(r.Context(), owner, name)
		if err != nil {
			render.NotFound(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Debugln("api: repository not found")
			return
		}

		user, err := users.Find(r.Context(), repo.UserID)
		if err != nil {
			render.NotFound(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Warnln("api: cannot find repository owner")
			return
		}

		remote, err := repoz.Find(r.Context(), user, repo.Slug)
		if err != nil {
			render.NotFound(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Warnln("api: remote repository not found")
			return
		}

		repo.Branch = remote.Branch
		repo.HTTPURL = remote.HTTPURL
		repo.Private = remote.Private
		repo.SSHURL = remote.SSHURL

		// the gitea and gogs repository endpoints do not
		// return the http url, so we need to ensure we do
		// not replace the existing value with a zero value.
		if remote.Link != "" {
			repo.Link = remote.Link
		}

		err = repos.Update(r.Context(), repo)
		if err != nil {
			render.InternalError(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Warnln("api: cannot chown repository")
			return
		}

		err = hooks.Create(r.Context(), user, repo)
		if err != nil {
			render.InternalError(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Debugln("api: cannot create or update hook")
			return
		}

		render.JSON(w, repo, 200)
	}
}
