# 1 "src/tools/xml_lexer.mll"
 
type error =
  | Illegal_character of char
  | Bad_entity of string
  | Unterminated of string
  | Tag_expected
  | Attribute_expected
  | Other of string

let error_string = function
  | Illegal_character c ->
      "illegal character '" ^ Char.escaped c ^ "'"
  | Bad_entity s ->
      "\"&" ^ s ^ ";\" is not a valid entity"
  | Unterminated s -> "unterminated " ^ s ^ " starts here"
  | Tag_expected -> "a tag was expected"
  | Attribute_expected -> "an attribute value was expected"
  | Other s -> s

exception Error of error * int

type token =
  | Tag of string * (string * string) list * bool
  | Chars of string
  | Endtag of string
  | EOF

let start_pos = ref 0
let reset_pos lexbuf =
  start_pos := Lexing.lexeme_start lexbuf

let raise_unterminated msg =
  raise (Error (Unterminated msg, !start_pos))

let buffer = Buffer.create 128
let reset_string lexbuf =
  reset_pos lexbuf ;
  Buffer.reset buffer

let strip_ws = ref true
let entities = ref [ "lt"  , "<"; 
		     "gt"  , ">"; 
		     "amp" , "&"; 
		     "apos", "'"; 
		     "quot", "\"" ]

let ws = [ ' '; '\009'; '\010'; '\012'; '\013' ]
let trim_ws s =
  let len = String.length s in
  let start = ref 0 in
  let stop = ref (len - 1) in
  while !start < len && List.mem s.[!start] ws do
    incr start 
  done ;
  while !stop >= !start && List.mem s.[!stop] ws do
    decr stop
  done ;
  if !start <> 0 || !stop <> len - 1
  then String.sub s !start (!stop - !start + 1)
  else s

# 64 "src/tools/xml_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\249\255\000\000\251\255\252\255\000\000\000\000\000\000\
    \001\000\255\255\002\000\000\000\000\000\000\000\001\000\254\255\
    \003\000\000\000\001\000\001\000\001\000\253\255\001\000\253\255\
    \254\255\002\000\029\000\253\255\107\000\255\255\254\255\194\000\
    \013\001\253\255\000\000\023\001\046\001\255\255\254\255\101\001\
    \101\001\254\255\205\001\072\001\254\255\255\255\010\000\234\001\
    \064\002\011\000\254\255\000\000\253\255\070\002\254\255\005\000\
    \075\002\080\002\086\000\253\255\254\255\255\255\030\000\253\255\
    \254\255\000\000\012\000\255\255\083\000\252\255\253\255\254\255\
    \255\255\165\000\252\255\253\255\254\255\255\255\031\000\253\255\
    \254\255\013\000\013\000\255\255\106\000\253\255\254\255\015\000\
    \255\255\206\001\047\002\251\255\252\255\253\255\128\000\136\000\
    \255\255\254\255\197\000\060\002\129\000\137\000\255\255\254\255\
    \253\255";
  Lexing.lex_backtrk =
   "\007\000\255\255\005\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\002\000\255\255\
    \255\255\000\000\255\255\255\255\002\000\255\255\255\255\255\255\
    \255\255\255\255\002\000\002\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\001\000\255\255\
    \003\000\255\255\255\255\000\000\255\255\255\255\255\255\000\000\
    \001\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\002\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\002\000\255\255\255\255\255\255\255\255\255\255\002\000\
    \255\255\255\255\005\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\003\000\255\255\255\255\255\255\255\255\
    \255\255";
  Lexing.lex_default =
   "\255\255\000\000\255\255\000\000\000\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\025\000\000\000\
    \000\000\025\000\027\000\000\000\255\255\000\000\000\000\255\255\
    \033\000\000\000\255\255\255\255\255\255\000\000\000\000\255\255\
    \041\000\000\000\255\255\044\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\000\000\054\000\000\000\255\255\
    \255\255\255\255\059\000\000\000\000\000\000\000\063\000\000\000\
    \000\000\255\255\255\255\000\000\069\000\000\000\000\000\000\000\
    \000\000\074\000\000\000\000\000\000\000\000\000\079\000\000\000\
    \000\000\255\255\255\255\000\000\085\000\000\000\000\000\255\255\
    \000\000\090\000\090\000\000\000\000\000\000\000\094\000\095\000\
    \000\000\000\000\099\000\099\000\100\000\101\000\000\000\000\000\
    \000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\051\000\051\000\000\000\051\000\051\000\055\000\055\000\
    \000\000\055\000\055\000\047\000\047\000\000\000\047\000\047\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\005\000\000\000\000\000\000\000\055\000\000\000\024\000\
    \255\255\000\000\047\000\000\000\000\000\008\000\009\000\003\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\082\000\038\000\002\000\255\255\255\255\004\000\
    \029\000\018\000\020\000\016\000\007\000\011\000\015\000\017\000\
    \045\000\052\000\067\000\083\000\081\000\088\000\000\000\010\000\
    \014\000\000\000\000\000\000\000\012\000\019\000\000\000\028\000\
    \000\000\013\000\000\000\006\000\021\000\066\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \061\000\071\000\072\000\065\000\028\000\060\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \031\000\031\000\000\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\030\000\097\000\
    \104\000\087\000\096\000\103\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\000\000\077\000\
    \000\000\000\000\031\000\076\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\000\000\101\000\
    \000\000\000\000\000\000\000\000\100\000\000\000\000\000\031\000\
    \031\000\000\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\030\000\000\000\000\000\
    \001\000\023\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\064\000\080\000\
    \000\000\031\000\102\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\046\000\046\000\070\000\046\000\046\000\255\255\000\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \046\000\037\000\086\000\000\000\000\000\000\000\000\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\000\000\000\000\000\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\000\000\000\000\
    \255\255\255\255\000\000\000\000\000\000\035\000\045\000\000\000\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\042\000\
    \038\000\000\000\000\000\000\000\000\000\075\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \000\000\000\000\000\000\000\000\042\000\102\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \095\000\000\000\000\000\047\000\047\000\094\000\047\000\047\000\
    \000\000\000\000\042\000\042\000\000\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \000\000\000\000\047\000\000\000\092\000\255\255\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \045\000\093\000\000\000\000\000\042\000\000\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \255\255\051\000\051\000\000\000\051\000\051\000\000\000\056\000\
    \056\000\255\255\056\000\056\000\057\000\057\000\255\255\057\000\
    \057\000\057\000\057\000\000\000\057\000\057\000\255\255\000\000\
    \051\000\000\000\000\000\255\255\000\000\255\255\056\000\000\000\
    \000\000\000\000\000\000\057\000\000\000\255\255\000\000\049\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\050\000\000\000\
    \000\000\000\000\000\000\055\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\255\255\000\000\000\000\055\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\091\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\051\000\051\000\255\255\051\000\051\000\055\000\055\000\
    \255\255\055\000\055\000\046\000\046\000\255\255\046\000\046\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \051\000\002\000\255\255\255\255\255\255\055\000\255\255\022\000\
    \025\000\255\255\046\000\255\255\255\255\005\000\008\000\002\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\081\000\034\000\000\000\022\000\025\000\002\000\
    \026\000\017\000\019\000\006\000\005\000\010\000\014\000\016\000\
    \046\000\049\000\066\000\082\000\078\000\087\000\255\255\007\000\
    \013\000\255\255\255\255\255\255\011\000\018\000\255\255\026\000\
    \255\255\012\000\255\255\005\000\020\000\065\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \058\000\068\000\068\000\062\000\026\000\058\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \028\000\028\000\255\255\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\094\000\
    \100\000\084\000\095\000\101\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\255\255\073\000\
    \255\255\255\255\028\000\073\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\255\255\098\000\
    \255\255\255\255\255\255\255\255\098\000\255\255\255\255\031\000\
    \031\000\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\255\255\255\255\
    \000\000\022\000\025\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\026\000\062\000\078\000\
    \255\255\031\000\098\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\043\000\043\000\068\000\043\000\043\000\058\000\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \043\000\036\000\084\000\255\255\255\255\255\255\255\255\036\000\
    \036\000\036\000\036\000\036\000\036\000\255\255\255\255\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\255\255\255\255\
    \094\000\100\000\255\255\255\255\255\255\032\000\043\000\255\255\
    \095\000\101\000\255\255\255\255\255\255\255\255\255\255\036\000\
    \036\000\036\000\036\000\036\000\036\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\040\000\
    \039\000\255\255\255\255\255\255\255\255\073\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \255\255\255\255\255\255\255\255\040\000\098\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \089\000\255\255\255\255\047\000\047\000\089\000\047\000\047\000\
    \255\255\255\255\042\000\042\000\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \255\255\255\255\047\000\255\255\089\000\032\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \047\000\089\000\255\255\255\255\042\000\255\255\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \043\000\048\000\048\000\255\255\048\000\048\000\255\255\053\000\
    \053\000\090\000\053\000\053\000\056\000\056\000\090\000\056\000\
    \056\000\057\000\057\000\255\255\057\000\057\000\099\000\255\255\
    \048\000\255\255\255\255\099\000\255\255\040\000\053\000\255\255\
    \255\255\255\255\255\255\056\000\255\255\090\000\255\255\048\000\
    \057\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\048\000\255\255\
    \255\255\255\255\255\255\053\000\255\255\255\255\255\255\255\255\
    \056\000\255\255\090\000\255\255\255\255\057\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\099\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\089\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\090\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\099\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\053\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec token lexbuf =
   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 68 "src/tools/xml_lexer.mll"
      ( reset_pos lexbuf; comment lexbuf; token lexbuf )
# 348 "src/tools/xml_lexer.ml"

  | 1 ->
# 70 "src/tools/xml_lexer.mll"
      ( reset_pos lexbuf; skip_doctype lexbuf; token lexbuf )
# 353 "src/tools/xml_lexer.ml"

  | 2 ->
# 72 "src/tools/xml_lexer.mll"
      ( reset_string lexbuf ;
	match cdata lexbuf with
	| "" -> token lexbuf
	| s  -> Chars (cdata lexbuf) )
# 361 "src/tools/xml_lexer.ml"

  | 3 ->
# 77 "src/tools/xml_lexer.mll"
      ( reset_pos lexbuf; skip_prolog_or_pi lexbuf; token lexbuf )
# 366 "src/tools/xml_lexer.ml"

  | 4 ->
# 79 "src/tools/xml_lexer.mll"
      ( let tag = tag_name lexbuf in 
        close_tag lexbuf ;
        Endtag tag )
# 373 "src/tools/xml_lexer.ml"

  | 5 ->
# 83 "src/tools/xml_lexer.mll"
      ( let tag = tag_name lexbuf in
        let attribs, closed = attributes lexbuf in
        Tag (tag, attribs, closed) )
# 380 "src/tools/xml_lexer.ml"

  | 6 ->
# 87 "src/tools/xml_lexer.mll"
      ( EOF )
# 385 "src/tools/xml_lexer.ml"

  | 7 ->
# 89 "src/tools/xml_lexer.mll"
      ( reset_string lexbuf;
	match chars lexbuf with
	| "" -> token lexbuf
	| s  -> Chars s )
# 393 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and chars lexbuf =
   __ocaml_lex_chars_rec lexbuf 22
and __ocaml_lex_chars_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 96 "src/tools/xml_lexer.mll"
      ( let len = lexbuf.Lexing.lex_curr_pos - lexbuf.Lexing.lex_start_pos in
	Buffer.add_subbytes buffer lexbuf.Lexing.lex_buffer lexbuf.Lexing.lex_start_pos len ;
        chars lexbuf )
# 407 "src/tools/xml_lexer.ml"

  | 1 ->
# 100 "src/tools/xml_lexer.mll"
      ( Buffer.add_string buffer (entity lexbuf) ;
	chars lexbuf )
# 413 "src/tools/xml_lexer.ml"

  | 2 ->
# 103 "src/tools/xml_lexer.mll"
      ( if !strip_ws
        then trim_ws (Buffer.contents buffer) 
        else Buffer.contents buffer )
# 420 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_chars_rec lexbuf __ocaml_lex_state

and entity lexbuf =
   __ocaml_lex_entity_rec lexbuf 26
and __ocaml_lex_entity_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 109 "src/tools/xml_lexer.mll"
      ( let s = Gutf8.from_unichar (charref lexbuf) in
        String.iter (fun c -> Printf.eprintf "%x" (Char.code c)) s;
        prerr_endline ""; s )
# 434 "src/tools/xml_lexer.ml"

  | 1 ->
# 113 "src/tools/xml_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let s = String.sub l 0 (String.length l - 1) in
        try List.assoc s !entities
        with Not_found -> raise (Error(Bad_entity s, Lexing.lexeme_start lexbuf)) )
# 442 "src/tools/xml_lexer.ml"

  | 2 ->
# 118 "src/tools/xml_lexer.mll"
      ( raise_unterminated "entity" )
# 447 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_entity_rec lexbuf __ocaml_lex_state

and charref lexbuf =
   __ocaml_lex_charref_rec lexbuf 32
and __ocaml_lex_charref_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 122 "src/tools/xml_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let v = String.sub l 1 (String.length l - 2) in
	int_of_string ("0x" ^ v) )
# 461 "src/tools/xml_lexer.ml"

  | 1 ->
# 126 "src/tools/xml_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let v = String.sub l 0 (String.length l - 1) in
	int_of_string v )
# 468 "src/tools/xml_lexer.ml"

  | 2 ->
# 130 "src/tools/xml_lexer.mll"
      ( raise (Error(Other "bad character reference", Lexing.lexeme_start lexbuf)) )
# 473 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_charref_rec lexbuf __ocaml_lex_state

and tag_name lexbuf =
   __ocaml_lex_tag_name_rec lexbuf 40
and __ocaml_lex_tag_name_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 134 "src/tools/xml_lexer.mll"
      ( Lexing.lexeme lexbuf )
# 485 "src/tools/xml_lexer.ml"

  | 1 ->
# 136 "src/tools/xml_lexer.mll"
      ( raise (Error(Tag_expected, Lexing.lexeme_start lexbuf)) )
# 490 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_tag_name_rec lexbuf __ocaml_lex_state

and close_tag lexbuf =
   __ocaml_lex_close_tag_rec lexbuf 43
and __ocaml_lex_close_tag_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 140 "src/tools/xml_lexer.mll"
      ( () )
# 502 "src/tools/xml_lexer.ml"

  | 1 ->
# 142 "src/tools/xml_lexer.mll"
      ( raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Lexing.lexeme_start lexbuf)) )
# 508 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_close_tag_rec lexbuf __ocaml_lex_state

and attributes lexbuf =
   __ocaml_lex_attributes_rec lexbuf 48
and __ocaml_lex_attributes_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 147 "src/tools/xml_lexer.mll"
      ( attributes lexbuf )
# 520 "src/tools/xml_lexer.ml"

  | 1 ->
# 149 "src/tools/xml_lexer.mll"
      ( [], false )
# 525 "src/tools/xml_lexer.ml"

  | 2 ->
# 151 "src/tools/xml_lexer.mll"
      ( [], true )
# 530 "src/tools/xml_lexer.ml"

  | 3 ->
# 153 "src/tools/xml_lexer.mll"
      ( let key = tag_name lexbuf in
        eq lexbuf ;
        let data = attribute_data lexbuf in
        let others, closed = attributes lexbuf in
        (key, data) :: others, closed )
# 539 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attributes_rec lexbuf __ocaml_lex_state

and eq lexbuf =
   __ocaml_lex_eq_rec lexbuf 53
and __ocaml_lex_eq_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 161 "src/tools/xml_lexer.mll"
      ( () )
# 551 "src/tools/xml_lexer.ml"

  | 1 ->
# 163 "src/tools/xml_lexer.mll"
      ( raise (Error(Attribute_expected, Lexing.lexeme_start lexbuf)) )
# 556 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_eq_rec lexbuf __ocaml_lex_state

and attribute_data lexbuf =
   __ocaml_lex_attribute_data_rec lexbuf 58
and __ocaml_lex_attribute_data_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 167 "src/tools/xml_lexer.mll"
      ( reset_string lexbuf; dq_string lexbuf )
# 568 "src/tools/xml_lexer.ml"

  | 1 ->
# 169 "src/tools/xml_lexer.mll"
      ( reset_string lexbuf;  q_string lexbuf )
# 573 "src/tools/xml_lexer.ml"

  | 2 ->
# 171 "src/tools/xml_lexer.mll"
      ( raise (Error(Attribute_expected, Lexing.lexeme_start lexbuf)) )
# 578 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attribute_data_rec lexbuf __ocaml_lex_state

and cdata lexbuf =
   __ocaml_lex_cdata_rec lexbuf 62
and __ocaml_lex_cdata_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 175 "src/tools/xml_lexer.mll"
      ( Buffer.contents buffer )
# 590 "src/tools/xml_lexer.ml"

  | 1 ->
# 177 "src/tools/xml_lexer.mll"
      ( raise_unterminated "CDATA" )
# 595 "src/tools/xml_lexer.ml"

  | 2 ->
# 179 "src/tools/xml_lexer.mll"
      ( Buffer.add_char buffer (Lexing.lexeme_char lexbuf 0) ;
	cdata lexbuf )
# 601 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_cdata_rec lexbuf __ocaml_lex_state

and q_string lexbuf =
   __ocaml_lex_q_string_rec lexbuf 68
and __ocaml_lex_q_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 184 "src/tools/xml_lexer.mll"
      ( Buffer.contents buffer )
# 613 "src/tools/xml_lexer.ml"

  | 1 ->
# 186 "src/tools/xml_lexer.mll"
      ( Buffer.add_string buffer (entity lexbuf); 
	q_string lexbuf )
# 619 "src/tools/xml_lexer.ml"

  | 2 ->
# 189 "src/tools/xml_lexer.mll"
      ( raise_unterminated "string" )
# 624 "src/tools/xml_lexer.ml"

  | 3 ->
# 191 "src/tools/xml_lexer.mll"
      ( Buffer.add_char buffer (Lexing.lexeme_char lexbuf 0); 
	q_string lexbuf )
# 630 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_q_string_rec lexbuf __ocaml_lex_state

and dq_string lexbuf =
   __ocaml_lex_dq_string_rec lexbuf 73
and __ocaml_lex_dq_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 196 "src/tools/xml_lexer.mll"
      ( Buffer.contents buffer )
# 642 "src/tools/xml_lexer.ml"

  | 1 ->
# 198 "src/tools/xml_lexer.mll"
      ( Buffer.add_string buffer (entity lexbuf); 
	dq_string lexbuf )
# 648 "src/tools/xml_lexer.ml"

  | 2 ->
# 201 "src/tools/xml_lexer.mll"
      ( raise_unterminated "string" )
# 653 "src/tools/xml_lexer.ml"

  | 3 ->
# 203 "src/tools/xml_lexer.mll"
      ( Buffer.add_char buffer (Lexing.lexeme_char lexbuf 0); 
	dq_string lexbuf )
# 659 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_dq_string_rec lexbuf __ocaml_lex_state

and comment lexbuf =
   __ocaml_lex_comment_rec lexbuf 78
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 208 "src/tools/xml_lexer.mll"
      ( () )
# 671 "src/tools/xml_lexer.ml"

  | 1 ->
# 210 "src/tools/xml_lexer.mll"
      ( raise_unterminated "comment" )
# 676 "src/tools/xml_lexer.ml"

  | 2 ->
# 212 "src/tools/xml_lexer.mll"
      ( comment lexbuf )
# 681 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and skip_prolog_or_pi lexbuf =
   __ocaml_lex_skip_prolog_or_pi_rec lexbuf 84
and __ocaml_lex_skip_prolog_or_pi_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 216 "src/tools/xml_lexer.mll"
      ( () )
# 693 "src/tools/xml_lexer.ml"

  | 1 ->
# 218 "src/tools/xml_lexer.mll"
      ( raise_unterminated "prolog or PI" )
# 698 "src/tools/xml_lexer.ml"

  | 2 ->
# 220 "src/tools/xml_lexer.mll"
      ( skip_prolog_or_pi lexbuf )
# 703 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_prolog_or_pi_rec lexbuf __ocaml_lex_state

and skip_doctype lexbuf =
   __ocaml_lex_skip_doctype_rec lexbuf 89
and __ocaml_lex_skip_doctype_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 224 "src/tools/xml_lexer.mll"
      ( skip_doctype lexbuf )
# 715 "src/tools/xml_lexer.ml"

  | 1 ->
# 226 "src/tools/xml_lexer.mll"
      ( skip_doctype lexbuf )
# 720 "src/tools/xml_lexer.ml"

  | 2 ->
# 228 "src/tools/xml_lexer.mll"
      ( skip_intsubset lexbuf; skip_doctype lexbuf )
# 725 "src/tools/xml_lexer.ml"

  | 3 ->
# 230 "src/tools/xml_lexer.mll"
      ( () )
# 730 "src/tools/xml_lexer.ml"

  | 4 ->
# 232 "src/tools/xml_lexer.mll"
      ( raise_unterminated "DOCTYPE" )
# 735 "src/tools/xml_lexer.ml"

  | 5 ->
# 234 "src/tools/xml_lexer.mll"
      ( skip_doctype lexbuf )
# 740 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_doctype_rec lexbuf __ocaml_lex_state

and skip_intsubset lexbuf =
   __ocaml_lex_skip_intsubset_rec lexbuf 98
and __ocaml_lex_skip_intsubset_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 238 "src/tools/xml_lexer.mll"
      ( () )
# 752 "src/tools/xml_lexer.ml"

  | 1 ->
# 240 "src/tools/xml_lexer.mll"
      ( skip_intsubset lexbuf )
# 757 "src/tools/xml_lexer.ml"

  | 2 ->
# 242 "src/tools/xml_lexer.mll"
      ( skip_intsubset lexbuf )
# 762 "src/tools/xml_lexer.ml"

  | 3 ->
# 244 "src/tools/xml_lexer.mll"
      ( skip_intsubset lexbuf )
# 767 "src/tools/xml_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_intsubset_rec lexbuf __ocaml_lex_state

;;

