use chrono::{DateTime, TimeDelta, Utc};
use glib::{Properties, subclass::*};
use gtk4::{Accessible, Buildable, CompositeTemplate, ConstraintTarget, Widget, prelude::*, subclass::prelude::*};
use libadwaita::{Bin, subclass::prelude::*};
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    static PROGRESS_TIMOUT: TimeDelta = TimeDelta::try_milliseconds(1500).unwrap();

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ProgressOverlay)]
    #[template(file = "data/resources/ui_templates/article_view/loading_progress.blp")]
    pub struct ProgressOverlay {
        #[property(get, set, nullable)]
        pub child: RefCell<Option<Widget>>,

        #[property(get, set = Self::set_progress)]
        pub progress: Cell<f64>,

        load_start_timestamp: RefCell<DateTime<Utc>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ProgressOverlay {
        const NAME: &'static str = "ProgressOverlay";
        type ParentType = Bin;
        type Type = super::ProgressOverlay;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ProgressOverlay {}

    impl WidgetImpl for ProgressOverlay {}

    impl BinImpl for ProgressOverlay {}

    #[gtk4::template_callbacks]
    impl ProgressOverlay {
        #[template_callback]
        fn is_visible(&self, progress: f64) -> bool {
            progress > 0.0 && progress < 1.0 && !self.hide_for_timeout()
        }

        fn hide_for_timeout(&self) -> bool {
            Utc::now() < *self.load_start_timestamp.borrow() + PROGRESS_TIMOUT
        }

        fn set_progress(&self, progress: f64) {
            if progress == 0.0 {
                self.load_start_timestamp.replace(Utc::now());
            }

            self.progress.set(progress);
        }
    }
}

glib::wrapper! {
    pub struct ProgressOverlay(ObjectSubclass<imp::ProgressOverlay>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ProgressOverlay {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}
