use crate::app::App;
use crate::error::NewsFlashGtkError;
use crate::i18n::i18n;
use glib::{Object, subclass::*};
use gtk4::{
    Accessible, Box, Buildable, Button, CompositeTemplate, ConstraintTarget, Stack, Widget, prelude::*,
    subclass::prelude::*,
};
use libadwaita::{Toast, ToastOverlay};
use news_flash::error::NewsFlashError;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate)]
    #[template(file = "data/resources/ui_templates/reset_page.blp")]
    pub struct ResetPage {
        #[template_child]
        pub reset_button: TemplateChild<Button>,
        #[template_child]
        pub reset_stack: TemplateChild<Stack>,
        #[template_child]
        pub toast_overlay: TemplateChild<ToastOverlay>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ResetPage {
        const NAME: &'static str = "ResetPage";
        type ParentType = gtk4::Box;
        type Type = super::ResetPage;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    impl ObjectImpl for ResetPage {}

    impl WidgetImpl for ResetPage {}

    impl BoxImpl for ResetPage {}

    #[gtk4::template_callbacks]
    impl ResetPage {
        #[template_callback]
        fn on_reset_clicked(&self) {
            self.reset_stack.set_visible_child_name("spinner");
            self.reset_button.set_sensitive(false);
            App::reset_account();
        }
    }
}

glib::wrapper! {
    pub struct ResetPage(ObjectSubclass<imp::ResetPage>)
        @extends Widget, Box,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ResetPage {
    fn default() -> Self {
        Object::new()
    }
}

impl ResetPage {
    pub fn reset(&self) {
        let imp = self.imp();

        imp.reset_stack.set_visible_child_name("label");
        imp.reset_button.set_sensitive(true);
    }

    pub fn error(&self, error: NewsFlashError) {
        self.reset();

        let message = i18n("Failed to reset account");
        let toast = Toast::new(&message);
        toast.set_button_label(Some(&i18n("details")));
        toast.set_action_name(Some("win.show-error-dialog"));

        App::default().set_newsflash_error(NewsFlashGtkError::NewsFlash {
            source: error,
            context: message,
        });

        self.imp().toast_overlay.add_toast(toast);
    }
}
