/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.openide.filesystems;
import junit.framework.*;
import org.netbeans.junit.*;

/**
 * A multi-threaded JUnit test case.
 *
*/
public class MultiThreadedTestCaseHid extends NbTestCase {
    /**
     * The threads that are executing.
     */
    private Thread threads[] = null;
    /**
     * The tests TestResult.*/
    private TestResult testResult = null;
    /**
     * Simple constructor.
     */

    public MultiThreadedTestCaseHid(String s) {
        super(s);
    }
    /**
     * Interrupt the running threads.
     */

    public void interruptThreads() {
        if(threads != null) {
            for(int i = 0;i < threads.length;i++) {
                //System.out.println("interrupted");
                threads[i].interrupt();
            }
        }
    }
    /**
     * Override run so we can squirrel away the test result.*/
    
    public void run(final TestResult result) {
        testResult = result;
        super.run(result);
        testResult = null;
    }
    /**
     * Run the test case threads.*/
    
    protected void runTestCaseRunnables (final TestCaseRunnable[] runnables, int ms) {
        if(runnables == null) {
            throw new IllegalArgumentException("runnables is null");
        }
        threads = new Thread[runnables.length];
        for(int i = 0;i < threads.length;i++) {
            threads[i] = new Thread(runnables[i]);
        }
        for(int i = 0;i < threads.length;i++) {
            threads[i].start();
        }
        try {
            long start = System.currentTimeMillis();
            for(int i = 0;i < threads.length;i++) {
                if (System.currentTimeMillis()-start > ms ) {
                    interruptThreads();
                    throw new InterruptedException ("Time ammount elapsed: probably deadlock.");
                }                
                threads[i].join((ms - (System.currentTimeMillis()-start)),0);
            }
        }
        catch(InterruptedException ignore) {
            handleException(ignore);
        }
        threads = null;
    }
    /**
     * Handle an exception. Since multiple threads won't have their
     * exceptions caught the threads must manually catch them and call
     * <code>handleException ()</code>.
     * @param t Exception to handle.*/
    
    private void handleException(final Throwable t) {
        synchronized(testResult) {
            if(t instanceof AssertionFailedError) {
                testResult.addFailure(this, (AssertionFailedError)t);
            }
            else {
                testResult.addError(this, t);
            }
        }
    }
    /**
     * A test case thread. Override runTestCase () and define
     * behaviour of test in there.*/
    protected abstract class TestCaseRunnable implements Runnable {
        /**
         * Override this to define the test*/
        
        public abstract void runTestCase()
                              throws Throwable;
        /**
         * Run the test in an environment where
         * we can handle the exceptions generated by the test method.*/
        
        public void run() {
            try {
                runTestCase();
            }
            catch(Throwable t) /* Any other exception we handle and then we interrupt the other threads.*/ {
                handleException(t);
                interruptThreads();
            }
        }
    }
}
